/* $Id: trigger.c,v 1.67 1999/12/15 22:07:28 yujik Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

#include "sema.h"

#define		TACH_XFER_TIMEOUT	1.0

#ifdef MICROSOFT_OS
#define		TACH_BUF_SIZE		E1432_TACH_RAW_SIZE/4
#else
#define		TACH_BUF_SIZE		E1432_TACH_RAW_SIZE
#endif

static long raw_tach_size;

/*
 *********************************************************************
 Set trigger parameters.  If mode is 'level', level1 is not used, but
 instead, a computed hysteresis is set (in set_trigger_level 0).
 Returns 0 if all ok, else return negative error number.
 Sets the channel active if trigger is turned on.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_trigger(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 state,
		  LONGSIZ32 delay, FLOATSIZ32 level0,
		  FLOATSIZ32 level1, SHORTSIZ16 slope,
		  SHORTSIZ16 mode)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_set_trigger"
		     "(0x%p, %d, %d, %d, %f, %f, %d, %d)\n",
		     hw, ID, state, delay, level0,
		     level1, slope, mode));

    error = e1432_set_trigger_channel(hw, ID, state);
    if (error)
	return error;

    error = e1432_set_trigger_delay(hw, ID, delay);
    if (error)
	return error;

    error = e1432_set_trigger_mode(hw, ID, mode);
    if (error)
	return error;

    error = e1432_set_trigger_slope(hw, ID, slope);
    if (error)
	return error;

    error = e1432_set_trigger_level(hw, ID, E1432_TRIGGER_LEVEL_LOWER,
				    level0);
    if (error)
	return error;

    error = e1432_set_trigger_level(hw, ID, E1432_TRIGGER_LEVEL_UPPER,
				    level1);
    if (error)
	return error;

    /* activate if trigger set on */
    if ( state == E1432_CHANNEL_ON )
    {
        error = e1432_set_active(hw, ID, E1432_CHANNEL_ON);
        if (error) return error;
    }

    return 0;
}


/*
 *********************************************************************
 Set trigger master state of modules;
 If ID == group:
     If <master> == OFF 
	 set all mods in group to OFF
	 group->trigmn = NULL;
     If <master> == MASTER
	 set all mods in group to SLAVE 
	 mod with 1st chan to MASTER,
	 group->trigmn = mod with 1st chan

 IF ID == channel:
    for all groups do:
	if group has channel ID
	    If <master> == OFF
	        if mod with chan is group->trigmn
	            trigmn = NULL
	    	    set all mods in group to off
	        else
	            set mod to OFF
	    If <master> == MASTER
	        set all mods to SLAVE
	        set mod with chan to MASTER
	        group->trigmn = mod with chan

 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_trigger_master(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 master)
{
    SHORTSIZ16 error, chanID;
    E1432_MODULE_LIST_NODE *mn, *mnchan;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 mod;
    int     mode;

    TRACE_PRINTF(0, ("e1432_set_trigger_master (0x%p, %d, %d)\n",
		     hw, ID, master));

    if(master != E1432_TRIGGER_MASTER_ON && master != E1432_TRIGGER_MASTER_OFF)
	return i1432_print_error(ERR1432_ILLEGAL_TRIGGER_MASTER);

    if (ID < 0) 	/* a group, set trigger master to first module */
    {
        gn = i1432_get_group_node(hw, ID);
	if(!gn) return i1432_print_error(ERR1432_NO_GROUP);
	gn->trigmn = NULL;
	mode = (master == E1432_TRIGGER_MASTER_OFF) ? 
			E1432_TRIGGER_MASTER_OFF : E1432_TRIGGER_SLAVE;
        
	/* Set all modules to either SLAVE or OFF */
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    error = i1432_set_mod(hw, i1432_get_chan_from_module(mn),
				  -1, E1432_CMD_SET_TRIGGER_MASTER,
				  mode);
	    if (error)
		return error;
	}

   	if(master == E1432_TRIGGER_MASTER_ON)
	{
	    if (gn->chanlist == NULL)
		/* No channels in group, don't try to set a trigger
		   master.  But don't error since we sometimes call
		   this function internally from modgroup.c
		   update_modlist(). */
		return 0;

	    chanID = gn->chanlist->chanID;

            error = i1432_set_mod(hw, chanID, -1,
				  E1432_CMD_SET_TRIGGER_MASTER, E1432_TRIGGER_MASTER_ON);
            if (error) return error;

	    error = i1432_get_module_from_chan(hw, chanID, &mn);
            if (error) return error;

	    gn->trigmn = mn;
	}
    }
    else	/* ID is a channel */
    {
	chanID = ID;

        /* get the master module associated with this channel */
        error = i1432_get_module_from_chan(hw, chanID, &mnchan);
        if (error) return error;

        /* clear all module flags */
        mn = i1432_mod_list;
        for (mod = 0; mod < i1432_mod_count; mod++, mn++)
	    mn->flag = 0;

	/* first find all groups with this module in them and set the pointer*/
        gn = i1432_group_list;
	while (gn != NULL)	/* for all groups */
        {
            for(mod = 0; mod < gn->modcount; mod++) /*for all module in group*/
            {
		gn->modlist[mod]->flag = 1;	/* flag this module */
                if(gn->modlist[mod] == mnchan)	/* if group has master module */
                {
	            /* set new trigger master module pointer */
	            if(master == E1432_TRIGGER_MASTER_ON) 
		        gn->trigmn = mnchan;
		    else
			gn->trigmn = NULL;
		}
	    }
	    gn = gn->next;
	}

	/* now go thru the groups again setting the hardware correctly */
        gn = i1432_group_list;
	while (gn != NULL)	/* for all groups */
	{
	    /* For all mods in group set right mode */
            for(mod = 0; mod < gn->modcount; mod++) 
            {
		mn = gn->modlist[mod];
		if(mn->flag)	/* if module not set yet */
		{
	            if(gn->trigmn != NULL)
	            {
		        mode = (mn == mnchan) ? E1432_TRIGGER_MASTER_ON :
			    E1432_TRIGGER_SLAVE;
		        mn->flag = 0;
		    }
		    else
		    {
		        mode = E1432_TRIGGER_MASTER_OFF;
		    }
		    
		    error = i1432_set_mod(hw, i1432_get_chan_from_module(mn),
					  -1, E1432_CMD_SET_TRIGGER_MASTER,
					  (LONGSIZ32) mode);
		    if (error) return error;
		}
   	    }

	    gn = gn->next;
	}
    }
    return 0;
}


/*
 *********************************************************************
 Get the a trigger index from a master module generating a trigger and
 send it to a slave module.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_send_trigger(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;
    LONGSIZ32 trigHi, trigLo, trigFlag, slaveReady;
    int mod, i, num, tindex;
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    float rpm1, rpm2;

    TRACE_PRINTF(0, ("e1432_send_trigger" "(0x%p, %d)\n", hw, ID));

    gn = i1432_get_group_node(hw, ID);
    if (!gn) return i1432_print_error(ERR1432_NO_GROUP);

#if 0
    if (gn->modcount < 2)	/* if only one module in group */
	return 0;
#endif
    
    if (gn->trigmn == NULL) 
	return i1432_print_error(ERR1432_NO_CHANNEL);

   
    /* make sure all slaves have processed last trigger before proceeding */
    for (mod = 0; mod < gn->modcount; mod++)
    {
	mn = gn->modlist[mod];
	if (gn->trigmn && mn != gn->trigmn)
	{
	    error = e1432_get_trigger_flag(hw, i1432_get_chan_from_module(mn),
					    			&slaveReady);
	    if (error)
		return error;
	    
	    if(!slaveReady)	/* module hasn't processed last trigger */
		return 0;
    	}
    }

    if (i1432_kludge_tach_flag == 2) /* Waiting for slave */
	i1432_kludge_tach_flag = 0; /* Slave tach ready */

    /* if the master has actually sent a trigger then tell all modules 
     * that slaves processed and accepted the last trigger */
    /* (This flag is cleared in e1432_init_measure_finish.) */
    if (i1432_trigger_sent)
    {
        for (mod = 0; mod < gn->modcount; mod++)
        {
            error = e1432_set_trigger_flag(hw, 
			i1432_get_chan_from_module(gn->modlist[mod]), -1);
            if (error) return error;
	}
    }
    i1432_trigger_sent = 0; /* Clear flag that indicates master sent trigger */

    /* see if master module has a new trigger to send */
    error = e1432_get_trigger_flag(hw, i1432_get_chan_from_module(gn->trigmn),
	    							&trigFlag);
    if (error) return error;

    /* if previous handshake still there or no triggers, return */
    if(trigFlag <= 0)
	return 0;

    /* At this point, we have verified that all slaves are ready, and
       the master has a new trigger.

       There are two possibilities.  One, a trigger was recently sent to
       the slaves but we haven't read any data from any modules yet,
       or two, a trigger was sent and we also read data.  In the first
       case, the kludge flag is zero, in the second it's one.

       In the first case, we don't want to use the new trigger in the
       master until we actually do read data from the slaves.
       Normally this takes care of itself, because the master won't
       find another trigger until the host reads out all its data.
       But if there are no active input channels in the master, then
       the master will move ahead before we read the data from the
       slaves, and we don't want that.  So exit here if the kludge
       flag is zero.

       This is potentially a problem if interrupts are being used and
       there is a delay before the slaves have all their data ready,
       because then we'll interrupt over and over until the slave data
       does become ready.  But similar problems exist anyway given the
       way our trigger passing was designed. */
    if (i1432_kludge_tach_flag != 1)
	return 0;

    /* find the first tach channel in the master module */
    tindex = 0;
    num = i1432_chan_count[E1432_CHAN_TYPE_TACH];
    for(i=0; i < num; i++)
	if(i1432_chan_list[E1432_CHAN_TYPE_TACH][i].mn == gn->trigmn)
	{
	    tindex = i1432_chan_list[E1432_CHAN_TYPE_TACH][i].index;
	    break;
	}
    if(tindex <= 0)
	return i1432_print_error(ERR1432_NO_TACH_BOARD);

    if (i1432_kludge_data_rpm_flag)
	return i1432_print_error(ERR1432_INTERNAL_LIBRARY_ERROR);
    i1432_kludge_data_rpm_id1 = (SHORTSIZ16) E1432_TACH_CHAN(i + 1);

    /* get data RPMs from master module */
    error = i1432_direct_read32_register(gn->trigmn, 
			E1432_DATA_RPM_CREG + tindex * E1432_CREG_OFFSET, 
			(LONGSIZ32 *)&rpm1);
    if (error)
        return error;
    error = i1432_direct_read32_register(gn->trigmn, 
	  		E1432_DATA_RPM_CREG + (tindex + 1) * E1432_CREG_OFFSET,
			(LONGSIZ32 *)&rpm2);
    if (error)
        return error;

    i1432_kludge_data_rpm1 = rpm1;
    i1432_kludge_data_rpm2 = rpm2;
    i1432_kludge_data_rpm_flag = 1;

    /* get tach trigger from master module */
    /* FOR PROPER OPERATION, ALWAYS DO trig_lo AFTER trig_hi */
    error = e1432_get_trig_hi(hw, i1432_get_chan_from_module(gn->trigmn),
	    							&trigHi);
    if (error) return error;

    error = e1432_get_trig_lo(hw, i1432_get_chan_from_module(gn->trigmn),
	    							&trigLo);
    if (error) return error;

    tindex = E1432_VIRTUAL_TACH_START;	/* tach index in slave modules */
    
    /* Send master rpm and trigger time to all slave modules */
    for (mod = 0; mod < gn->modcount; mod++)
    {
	/* Do one module */
	mn = gn->modlist[mod];
	if (gn->trigmn && mn != gn->trigmn)
	{
	    error = i1432_direct_write32_register(mn, 
			E1432_DATA_RPM_CREG + tindex * E1432_CREG_OFFSET, 
			*((LONGSIZ32 *)&rpm1));
    	    if (error) return error;
	    
	    error = i1432_direct_write32_register(mn, 
			E1432_DATA_RPM_CREG + (tindex + 1) * E1432_CREG_OFFSET,
			*((LONGSIZ32 *)&rpm2));
    	    if (error) return error;

	    error = e1432_set_trig_hi(hw, i1432_get_chan_from_module(mn),
					    			trigHi);
	    if (error) return error;

	    error = e1432_set_trig_lo(hw, i1432_get_chan_from_module(mn),
					    			trigLo);
	    if (error) return error;
	}
    }

    /* tell master module trigger has been sent to slaves */
    i1432_trigger_sent = 1;
    error = e1432_set_trigger_flag(hw, i1432_get_chan_from_module(gn->trigmn), 
									0);
    if (error) return error;

    if (i1432_kludge_tach_flag != 1) /* Need new trigger */
	return i1432_print_error(ERR1432_INTERNAL_LIBRARY_ERROR);
    i1432_kludge_tach_flag = 2;	/* Waiting for slave */

    return 0;
}


/*
 *********************************************************************
 Do the raw tach time transfer by going directly to the Y memory.  The
 substrate 96002 doesn't take any part in the transfer, and we must
 send a command when we're done to let the 96002 know that we're
 done.  
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16 EXPORT
e1432_get_raw_tachs(E1432ID hw, SHORTSIZ16 ID, unsigned long *lbuf,
     				LONGSIZ32 size,	LONGSIZ32 *count)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 val, error;
    LONGSIZ32 start, length, numTachs;
    long    index, mask;
    int     cindex;

    if (ID < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);

    /* get module for this channel */
    error = i1432_get_module_from_chan(hw, ID, &mn);
    if (error)
	return error;

    if(((ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT) !=
						E1432_CHAN_TYPE_TACH) 
	return i1432_print_error(ERR1432_NOT_TACH_CHANNEL);

    error = e1432_read_register(hw, ID, E1432_IRQ_STATUS2_REG, &val);
    if (error)
	return error;

    /*
     * The following two lines are added to detect TACH BUFFER OVERFLOW.
     *  The change is made to fix a tach related bug in continuous mode
     *  that is reported by MBBM.
     *  Dec 15, 99,  Yuji Kobayashi
     */
    if ((val & E1432_STATUS2_TACH_OVERFLOW) != 0)
	return i1432_print_error(ERR1432_TACH_BUFFER_OVERFLOW);

    error = e1432_read32_register(hw, ID, 
				E1432_RAW_TACH_SIZE_REG, &raw_tach_size);
    if (error)
        return error;
    mask = raw_tach_size - 1;

    cindex = i1432_get_chan_index(hw, ID);

    error = e1432_read32_register(hw, ID, 
	E1432_RAW_TACH_PTR_CREG + cindex * E1432_CREG_OFFSET, &start);
    if (error)
        return error;

#if 0
#ifdef HAVE_VTL
    /* if using VISA/VTL and a 256K a24 board, then the tach window is */
    /* 20000 above where it is on the a24 1M board */
    if(mn->a24_256k)
	start  += 0x20000;
#endif
#endif
    
    error = e1432_read32_register(hw, ID, 
	E1432_RAW_TACH_COUNT_CREG + cindex * E1432_CREG_OFFSET, &numTachs);
    if (error)
        return error;

    if(numTachs > raw_tach_size)
	return i1432_print_error(ERR1432_TACH_BUFFER_OVERFLOW);

    if(numTachs > size)
	numTachs = size;

    *count = 0;

    while (numTachs > 0)
    {
	length = numTachs;

	index = start & mask;		/* index into tach buffer in subst */
	if(index + length > raw_tach_size)
	{
	    length = raw_tach_size - index;   /* do to end of buffer */
	}

	/* cast VXI address to avoid DOS segment math */
	error = i1432_xfer_block(mn, (unsigned long *) (start * 4L),
				 lbuf, length, 0, 1);
	if (error)
	    return error;

#if SWAP
	if (!mn->d32)
	{
	    /* The data is 32-bit data.  If the data was transferred
	       (and byte swapped) in 16-bit chunks, we need to swap
	       the data now to fix it. */
	    error = i1432_two_four_swap((char *) lbuf, length * 4);
	    if (error)
		return error;
	}
#endif

	lbuf += length;
	numTachs -= length;
        *count += length;
	start += length;
	if(index + length == raw_tach_size)
	    start -= raw_tach_size;	/* address to start of buffer */
    }

    /* No need to handshake with the module if there were no tachs.
       But if there were, we need to tell the module's firmware how
       many we read. */
    if (*count != 0)
    {
	error = i1432_write_cmd2(hw, ID, E1432_CCMD_RAW_TACHS_DONE, 
				 cindex, *count);
	if (error)
	    return error;
    }

    return 0;
}


/*
 *********************************************************************
 Do the raw tach time transfer by going directly to the Y memory.  The
 substrate 96002 doesn't take any part in the transfer, and we must
 send a command when we're done to let the 96002 know that we're
 done.  
 *********************************************************************
 */
/*ARGSUSED*/
static SHORTSIZ16
e1432_send_raw_tachs(E1432ID hw, E1432_MODULE_LIST_NODE *mn, int tachIndex,
				unsigned long *lbuf, LONGSIZ32 numTachs)
{
    SHORTSIZ16 error;
    LONGSIZ32 start, cindex;
    LONGSIZ32 count;
#if 0
    FLOATSIZ64 startTime, currentTime;
#endif
    int mask = (int) raw_tach_size - 1;
    long index;
    long length;

    cindex = E1432_VIRTUAL_TACH_START + tachIndex;
    error = i1432_direct_read32_register(mn, E1432_RAW_TACH_PTR_CREG +
				cindex * E1432_CREG_OFFSET, &start);
    if (error)
        return error;

#if 0
    count = 0;
    i1432_get_time(&startTime);

    while (count < numTachs)
    {
        error = i1432_direct_read32_register(mn, E1432_RAW_TACH_COUNT_CREG +
				cindex * E1432_CREG_OFFSET, &count);
        if (error)
            return error;

	i1432_get_time(&currentTime);
	if ((currentTime - startTime) > TACH_XFER_TIMEOUT)
	{
	    return i1432_print_error(ERR1432_TACH_XFER_TIMEOUT);
	}
    }
#endif

#if SWAP
    if (!mn->d32)
    {
	/* The data is 32-bit data.  If the data will be transferred
	   (and byte swapped) in 16-bit chunks, we need to swap the
	   data now to fix it. */
	error = i1432_two_four_swap((char *) lbuf, numTachs * 4);
	if (error)
	    return error;
    }
#endif

    count = numTachs;
    while(count > 0)
    {
	length = count;
        index = start & mask;		/* index into tach buffer in subst */
        if(index + length > raw_tach_size)
        {
            length = raw_tach_size - index;   /* do to end of buffer */
        }

	error = i1432_xfer_block(mn, lbuf, (unsigned long *) (start * 4L),
				 length, 0, 0);
	if (error)
	    return error;

	lbuf += length;
	count -= length;
	start += length;
	if(index + length == raw_tach_size)
	    start -=  raw_tach_size;	/* address to start of buffer */
    }
    return 0;
}


/*
 *********************************************************************
 Get the raw tach times from the master module and distribute them to the
 slave modules.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_send_tachs(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 tachID,
			unsigned long *buf, int bufSize, LONGSIZ32 *actualCount)
{
    SHORTSIZ16 error;
    LONGSIZ32 count = 0, avail, temp;
    unsigned long *ptr;
    int     mod, tachIndex, cindex;
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    unsigned long tachBuf[TACH_BUF_SIZE];
    int size;

    TRACE_PRINTF(0, ("e1432_send_tachs" "(0x%p, %d, %d, 0x%p, 0x%p)\n", 
			hw, groupID, tachID, buf, actualCount));

    *actualCount = 0;

    gn = i1432_get_group_node(hw, groupID);
    if (!gn) return i1432_print_error(ERR1432_NO_GROUP);

    if(((tachID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT) !=
						E1432_CHAN_TYPE_TACH) 
	return i1432_print_error(ERR1432_NOT_TACH_CHANNEL);

    if(buf != NULL)
    {
	ptr = buf;
	size = bufSize;
    }
    else
    {
	ptr = tachBuf;
	size = TACH_BUF_SIZE;
    }

    /* get raw tachs from master module */
    error = e1432_get_raw_tachs(hw, tachID, ptr, size, &count);
    if (error) return error;

    if(count == 0) 
	return 0;

    *actualCount = count;
    
    if (gn->modcount < 2)	/* if only one module in group */
	return 0;

    if (gn->trigmn == NULL) 
	return i1432_print_error(ERR1432_NO_CHANNEL);

    /* PROBLEM: this little shortcut assumes that tach channels have the last
     * two indices in a module .... ugly, but it works when you're in a
     * hurry to get this code out. 
     */
    tachIndex = i1432_get_chan_index(hw, tachID) -
    				(gn->trigmn->nchan - E1432_TACH_CHANS);

    cindex = E1432_VIRTUAL_TACH_START + tachIndex;

    avail = count;
    /* find the maximum number of tachs that can be sent to all modules */
    for (mod = 0; mod < gn->modcount; mod++)
    {
        /* Do one module */
	mn = gn->modlist[mod];
	if (gn->trigmn && mn != gn->trigmn && mn->input_chans > 0)
	{
    	    error = i1432_direct_read32_register(mn, E1432_RAW_TACH_COUNT_CREG +
				cindex * E1432_CREG_OFFSET, &temp);
	    if (error)
	        return error;
	    if(temp < avail)
		avail = temp;
	}
    }

    if(avail != count)	/* correct the raw tach count in master */
    {
        cindex = i1432_get_chan_index(hw, tachID);
        error = i1432_write_cmd2(hw, tachID, E1432_CCMD_RAW_TACHS_DONE, 
							cindex, avail - count);
        if(error) return error;
    }

    cindex = E1432_VIRTUAL_TACH_START + tachIndex; 
    /* Send tachs to all slave modules and tell them how many */
    for (mod = 0; mod < gn->modcount; mod++)
    {
        /* Do one module */
	mn = gn->modlist[mod];
	if (gn->trigmn && mn != gn->trigmn && mn->input_chans > 0)
	{
	    error = e1432_send_raw_tachs(hw, mn, tachIndex, ptr, avail);
	    if (error)
	        return error;
                
	    error = i1432_write_cmd2(hw, i1432_get_chan_from_module(mn), 
				E1432_CCMD_RAW_TACHS_DONE, cindex, avail);
	    if(error) return error;

	}
    }

    *actualCount = avail;
    
    return 0;
}              
